' Sudoku for CMM2
' Rev 1.1.0 William M Leue 11/8/2020
' Rev 1.2.0 11/10/2020 - added solve clock

option default integer
option base 1

' Constants
const ORDER = 3
const SIZE = ORDER*ORDER
const CSIZE = 42
const UP = 128
const DOWN = 129
const LEFT = 130
const RIGHT = 131
const ENTER = 13
const ZERO = 48
const ONE = 49
const TWO = 50
const THREE = 51
const FOUR = 52
const FIVE = 53
const SIX = 54
const SEVEN = 55
const EIGHT = 56
const NINE = 57
const SPACE = 32
const H = 72 : const HL = 104
const M = 77 : const ML = 109
const Q = 81 : const QL = 113
const R = 82 : const RL = 114
const S = 83 : const SL = 115
const U = 85 : const UL = 117
const V = 86 : const VL = 118

const FCOLOR = RGB(100, 40, 40)
const CCOLOR = RGB(255, 255, 150)
const OCOLOR = RGB(BLUE)
const UCOLOR = RGB(BLACK)

const MLOAD  = 1
const MSAVE  = 2
const MSOLVE = 3
const MUNEW  = 4
const MHELP  = 5
const MQUIT  = 6
const NMENUS = 6
const MENU   = 10
const UNDO   = 8
const MVALS  = 9

const MENUX = 150
const MENUY = 100
const MENUW = 500
const MENUH = 40

const HDX = 7
const HDY = 10

const STACK_SIZE = 100
const CONFLICT = 100000

const CLOCKX = 630
const CLOCKY = 130
const KCOLOR = RGB(CYAN)

' only number values currently implemented
const SNUMBERS = 1
const SLETTERS = 2
const SFIGURES = 3
const SCOLORS  = 4
const NSYMBOLS = 4

const WCLOCKX = 70
const WCLOCKR = 30

' Globals
dim board(SIZE, SIZE)
dim have_puzzle = 0
dim prev_row = 0
dim prev_col = 0
dim gmode = 0
dim hmargin = 0
dim vmargin = 0
dim updated = 0
dim prev_menu = 0
dim show_hints = 0
dim pname$ = ""
dim menu_start_choice = 0
dim stack(STACK_SIZE)
dim sptr = 1
dim symbols = SNUMBERS
dim debug = 0
dim clockRunning = 0
dim clockSecs = 0
dim clockMins = 0
dim clockHrs = 0

' Main Program
SetGraphics
'open "debug.txt" for output as #1
menu_start_choice = MLOAD
do
  DrawMenuScreen
loop
end

' Set the Graphics mode
sub SetGraphics
  mode 1,8
end sub

' Show the Menu of available commands
sub DrawMenuScreen
  local x, y, i, cmd, choice

  cls
  x = MENUX
  y = MENUY
  text MM.HRES\2, 20, "Sudoku Command Menu", "CT", 5
  text MM.HRES\2, 60, "Use up and down arrows to navigate, Enter to execute", "CT"
  for i = 1 to NMENUS
    box x, y, MENUW, MENUH
    select case i
      case MLOAD
        text x+10, y+15, "Load a Sudoku puzzle from a file"
      case MSAVE
        text x+10, y+15, "Save the Suduku puzzle to a file"
      case MSOLVE
        text x+10, y+15, "Solve the Sudoku puzzle manually"
      case MUNEW
        text x+10, y+15, "Enter new a Sudoku manually"
      case MHELP
        text x+10, y+15, "Show Instructions"
      case MQUIT
        text x+10, y+15, "Quit the game"
    end select
    y = y + MENUH+10
  next i

  choice = menu_start_choice
  HiliteMenu choice
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(z$)
    select case cmd
      case UP
        choice = choice-1
        if choice < 1 then choice = 1
        HiliteMenu choice
      case DOWN
        choice = choice+1
        if choice > NMENUS then choice = NMENUS
        HiliteMenu choice
      case ENTER
        exit do
    end select
  loop  
  select case choice
    case MLOAD
      LoadPuzzle            
    case MSAVE
      SavePuzzle
    case MSOLVE
      UserSolvePuzzle
    case MUNEW
      UserCreatePuzzle
    case MHELP
      ShowInstructions
    case MQUIT
      'close #1
      cls
      end
    end select
end sub

' Hilite the current Menu Line
sub HiliteMenu which
  local xv(4), yv(4)
  local c

  if prev_menu > 0 then
    xv(1) = MENUX-10        : yv(1) = MENUY+(prev_menu-1)*(MENUH+10)+MENUH\2
    xv(2) = xv(1)-20        : yv(2) = yv(1) - MENUH\2
    xv(3) = xv(2)           : yv(3) = yv(1) + MENUH\2
    xv(4) = xv(1)           : yv(4) = yv(1)
    polygon 4, xv(), yv(),  RGB(BLACK), RGB(BLACK)
  end if
  xv(1) = MENUX-10        : yv(1) = MENUY+(which-1)*(MENUH+10)+MENUH\2
  xv(2) = xv(1)-20        : yv(2) = yv(1) - MENUH\2
  xv(3) = xv(2)           : yv(3) = yv(1) + MENUH\2
  xv(4) = xv(1)           : yv(4) = yv(1)
  polygon 4, xv(), yv(),  RGB(RED), RGB(RED)
  prev_menu = which
end sub

' Get user input for instant menu items
sub getUserInput z$, cmd
  local n

  n = asc(z$)
  select case n
    case H
      cmd = MHINT
    case HL
      cmd = MHINT
    case M
      cmd = MENU
    case ML
      cmd = MENU
    case R
      cmd = MRESET
    case RL
      cmd = MRESET
    case S
      cmd = MSOLVE
    case SL
      cmd = MSOLVE
    case U
      cmd = UNDO
    case UL
      cmd = UNDO
    case Q
      cmd = MQUIT
    case QL
      cmd = MQUIT
    case else
      ' ignore
  end select
end sub
      
' Load a puzzle from a '.sku' file
sub LoadPuzzle
  local buf$, v$
  local row, col, ok
  
  cls
  ZeroPuzzle 2
  pname$ = ""
  v$ = INKEY$
  do
    ok = 1
    input "Enter filename (extension will be added automatically): ", pname$
    if instr(z$, ".sku") = 0 then
      pname$ = pname$ + ".sku"
    end if
    on error skip 1
    open pname$ for input as #2    
    if MM.ERRNO > 0 then
      print "Sorry, that file was not found - please try again"
      print ""
      ok = 0
    end if
  loop until ok = 1
  for row = 1 to SIZE
    line input #2, buf$
    buf$ = buf$ + " "
    for col = 1 to SIZE
      v$ = FIELD$(buf$, col, " ")
      board(row, col) = val(v$)
    next col
  next row
  close #2 
  have_puzzle = 1
  UserSolvePuzzle  
end sub

' Save the current puzzle to a '.sku' file
sub SavePuzzle
  local z$
  local row, col, ok
  
  cls
  z$ = INKEY$
  do
    ok = 1
    input "Enter filename (extension will be added automatically): ", pname$
    if instr(z$, ".sku") = 0 then
      pname$ = pname$ + ".sku"
    end if
    on error skip 1
    open pname$ for input as #2
    if MM.ERRNO = 0 then
      close #2
      input "That file already exists. Overwrite? (Y,N): ", z$
      if LEFT$(UCASE$(z$), 1) = "N" then
        ok = 0
      end if
    end if    
  loop until ok = 1
  open pname$ for output as #2
  for row = 1 to SIZE
    for col = 1 to SIZE
      print #2, str$(board(row, col)) + " ";
    next col
    print #2, ""
  next row
  close #2
end sub

' Let the User Interactively create a new Sudoku puzzle
sub UserCreatePuzzle
  local z$
  local cmd, row, col, done

  cls
  done = 0
  ZeroPuzzle 2
  gmode = MUCRT
  text MM.HRES\2, 20, "User Puzzle Creation", "CT", 3
  text MM.HRES\2, 50, "Use the arrow keys to navigate around the board", "CT"
  text MM.HRES\2, 70, "Type the number for a cell, Press Enter when done", "CT"
  row = 1 : col = 1
  updated = 1
  DrawPuzzle row, col
  HiliteCell row, col
  updated = 0
  z$ = INKEY$
  do
    z$ = ""
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(z$)
    select case cmd
      case UP
        row = row-1
        if row < 1 then row = SIZE
        HiliteCell row, col
      case DOWN
        row = row+1
        if row > SIZE then row = 1
        HiliteCell row, col
      case LEFT
        col = col-1
        if col < 1 then col = SIZE
        HiliteCell row, col
      case RIGHT
        col = col+1
        if col > SIZE then col = 1
        HiliteCell row, col
      case ZERO to NINE
        if not done then
          num = cmd-ZERO
          pushMove row, col, num
          updated = 1
          DrawPuzzle row, col
          updated = 0
        end if
      case SPACE
        if not done then
          num = 0
          pushMove row, col, num
          DrawPuzzle row, col
        end if
      case ENTER
        have_puzzle = 1
        done = 1
        menu_start_choice = MSAVE
        text MM.HRES\2, 550, "Puzzle Creation Complete", "CT", 5
        MakeCellsPermanent
        clearMoveStack
        exit sub
      case else
        GetUserInput z$, icmd
        select case icmd
          case MHINT
            show_hints = 1-show_hints
            updated = 1
            DrawPuzzle row, col
            updated = 0
          case UNDO
            if not done then
              popMove
              updated = 1
              DrawPuzzle
              updated = 0
            end if
          case MQUIT
            'close #1
            cls
            end
          case MENU
            MakeCellsPermanent
            exit do
          case else
            ' ignore
        end select     
    end select
  loop
end sub

' make non-zero cells permanent by adding 100
sub MakeCellsPermanent
  local row, col, cvalue
  for row = 1 to SIZE
    for col = 1 to SIZE
      cvalue = board(row, col)
      if cvalue > 0 then
        board(row, col) = 100 + cvalue
      end if
    next col
  next row
end sub

' push a move onto the Undo stack
' Moves are encoded as row*10000 + col*1000 + num
sub pushMove row, col, num
  local cmark
  stack(sptr) = row*10000 + col*1000 + board(row, col)
  sptr = sptr+1
  cmark = 0
  if CheckConflict(row, col, num) then cmark = CONFLICT
  board(row, col) = cmark + num
end sub

' pop a move from the Undo stack
sub popMove
  local row, col, sv, num, cmark
  if sptr > 1 then
    sptr = sptr-1
    sv = stack(sptr)
    if sv >= CONFLICT then
      sv = sv - CONFLICT
    end if
    row = sv\10000
    col = (sv - row*10000)\1000
    num = (sv - row*10000 - col*1000)
    board(row, col) = num
end sub

' Clear the Undo stack
sub clearMoveStack
  sptr = 1
end sub

' Zero a puzzle 1: zero non-original cells 2: zero all cells
' Also reset the solve clock.
sub ZeroPuzzle which
  local row, col, val
  for row = 1 to SIZE
    for col = 1 to SIZE
      val = board(row, col)
      if which = 1 then
        if board(row, col) < 100 then
          board(row, col) = 0
        end if
      else
        board(row, col) = 0
      end if
    next col
  next row
  clockSecs = 0 : clockMins = 0 : clockHrs = 0
end sub

' Do trivial solving of cells and return number of cells solved
' The computer will call this after every major step in solving a puzzle
sub DoTrivialSolving num
  local c1, c2, tp
  do
    c1 = 0 : c2 = 0 : tp = 0
    do
      FindResidualNumbers c1
      DrawWaitClock 1
      FindSubgridNumbers c2
      DrawWaitClock 1
    loop until c1+c2 = 0
    if CheckWin() then exit do
    FindUniqueNumbers c3
    DrawWaitClock 1
    tp = tp + c1 + c2 + c3
  loop until c1+c2+c3 = 0
  DrawWaitClock 0
end sub    

' Let the User Interactively solve the current Sudoku puzzle
sub UserSolvePuzzle
  local z$
  local cmd, row, col, icmd, num, solved

  cls
  text MM.HRES\2, 20, "Solve puzzle '" + pname$ + "' Manually", "CT", 3
  text MM.HRES\2, 50, "Use the arrow keys to navigate around the board", "CT"
  text MM.HRES\2, 70, "Type the number for a cell or 0 (zero) to clear it", "CT"
  text MM.HRES\2, 550, "Instant Commands: H - toggle hints, U - undo", "CT"
  text MM.HRES\2, 565, "S - Solve Trivial Cells, M - return to menu", "CT"

  row = 1 : col = 1
  updated = 1
  DrawPuzzle row, col
  HiliteCell row, col
  updated = 0
  solved = 0
  clockRunning = 1
  settick 1000, UpdateClock

  z$ = INKEY$
  do
    z$ = ""
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(z$)
    select case cmd
      case UP
        row = row-1
        if row < 1 then row = SIZE
        HiliteCell row, col
      case DOWN
        row = row+1
        if row > SIZE then row = 1
        HiliteCell row, col
      case LEFT
        col = col-1
        if col < 1 then col = SIZE
        HiliteCell row, col
      case RIGHT
        col = col+1
        if col > SIZE then col = 1
        HiliteCell row, col
      case ZERO to NINE
        if not solved then
          cvalue = board(row, col)
          if cvalue >= 100000 then
            cvalue = cvalue - 100000
          end if
          if cvalue < 100 then
            num = cmd-ZERO
            pushMove row, col, num
            updated = 1
            DrawPuzzle row, col
            updated = 0
            if CheckWin() then 
              AnnounceSolved
              solved = 1
              ClearMoveStack
            end if
          end if
        end if
      case SPACE
        if not solved then
          num = 0
          pushMove row, col, num
          DrawPuzzle row, col
        end if
      case else
        GetUserInput z$, icmd
        select case icmd
          case MHINT
            show_hints = 1-show_hints
            updated = 1
            DrawPuzzle row, col
            updated = 0
          case UNDO
            if sptr > 1 then
              popMove
              updated = 1
              DrawPuzzle
              updated = 0
            end if
          case MSOLVE
            DoTrivialSolving num
            updated = 1
            DrawPuzzle row, col
            updated = 0
            if CheckWin() then 
              AnnounceSolved
              solved = 1
              clockRunning = 0
              settick 0, 1
              clearMoveStack
            end if
          case MVALS
            'symbols = symbols+1
            'if symbols > NSYMBOLS then symbols = SNUMBERS
            'updated = 1
            'DrawPuzzle row, col
            'updated = 0
          case MQUIT
            'close #1
            cls
            end
          case MENU
            clockRunning = 0
            settick 0, 1
            exit do
          case else
            ' ignore
        end select     
      end select
  loop
end sub

' Display a clock showing HH:MM:SS elapsed time for solving
' a puzzle.
sub UpdateClock
  local hh$, mm$, ss$
  if clockRunning then
    clockSecs = clockSecs+1
    if clockSecs = 60 then
      clockMins = clockMins+1
      clockSecs = 0
      if clockMins = 60 then
        clockHrs = clockHrs+1
        clockMins = 0
      end if
    end if
    hh$ = format$(clockHrs, "%02g")
    mm$ = format$(clockMins, "%02g")
    ss$ = format$(clockSecs, "%02g")
    text CLOCKX, CLOCKY, "         ", "LT", 3
    text CLOCKX, CLOCKY, hh$ + ":" + mm$ + ":" + ss$, "LT", 3, 1, KCOLOR
  end if
end sub

sub AnnounceSolved
  local y
  y = vmargin + SIZE*CSIZE + 20
  text MM.HRES\2, y, "Puzzle Solved!", "CT", 5
end sub

' Draw the Sudoku Puzzle.
' If hr and hc point to a cell, it will be hilited in green
sub DrawPuzzle hr, hc
  local row, col, x, y, w, h, x1, y1, x2, y2, lw, off
  local cx, cy, cval, n, c, cmark

  ' locate the board
  w = CSIZE*SIZE
  h = w
  x = MM.HRES\2 - w\2
  y = MM.VRES\2 - h\2
  hmargin = x
  vmargin = y

  ' frame
  box x-7, y-7, w+14, h+14, 7, FCOLOR, CCOLOR
  ' cells
  for row = 1 to SIZE-1
    lw = 1
    off = 0
    if row = 3 or row = 6 then
      lw = 5
      off = 2
    end if
    y1 = y + row*CSIZE - off
    y2 = y1
    x1 = x
    x2 = x + w
    line x1, y1, x2, y2, lw, RGB(BLACK)
  next row
  for col = 1 to SIZE-1
    lw = 1
    off = 0
    if col = 3 or col = 6 then
      lw = 5
      off = 2
    end if
    y1 = y
    y2 = y + h
    x1 = x + col*CSIZE - off
    x2 = x1
    line x1, y1, x2, y2, lw, RGB(BLACK)
  next col    
  ' numbers
  ' values >= 100 are original puzzle values
  ' values < 100 are user or computer entries while solving
  ' original values are printed in blue, other values in black
  ' the conflict marker is 1000000
  for row = 1 to SIZE
    for col = 1 to SIZE
      cval = board(row, col)
      cmark = 0
      if cval >= CONFLICT then
        cmark = 1
        cval = cval - CONFLICT
      end if
      n = cval mod 100
      cx = x + (col-1)*CSIZE + CSIZE\2
      cy = y + (row-1)*CSIZE + CSIZE\2
      c = UCOLOR
      if cval >= 100 then c = OCOLOR
      if n > 0 then
        'if cmark then
        '  bx = hmargin + (col-1)*CSIZE
        '  by = vmargin + (row-1)*CSIZE
        '  box bx+2, by+2, CSIZE-4, CSIZE-4,, RGB(RED)
        'end if
        text cx, cy+3, str$(n), "CM", 5,, c, CCOLOR
      else      
        text cx, cy+3, " ", "CM", 5,, c, CCOLOR
        if show_hints and updated then
          DrawHints row, col
        end if
      end if
    next col
  next row
  if hr > 0 and hc > 0 then HiliteCell hr, hc
end sub

' Hilite the currently-selected cell with a green outline
sub HiliteCell row, col
  local c, cvalue, cmark
  c = RGB(GREEN)
  cvalue = board(row, col)
  if cvalue >= CONFLICT then
    c = RGB(RED)
  end if
  if row > 0 and col > 0 then
    if prev_row > 0 and prev_col > 0 then
    cx = hmargin + (prev_col-1)*CSIZE+2
    cy = vmargin + (prev_row-1)*CSIZE+2
      box cx, cy, CSIZE-4, CSIZE-4, 1, CCOLOR
    end if
    cx = hmargin + (col-1)*CSIZE+2
    cy = vmargin + (row-1)*CSIZE+2
    box cx, cy, CSIZE-4, CSIZE-4, 1, c
    prev_row = row : prev_col = col
  end if
end sub

' Draw the mini-hints in a puzzle cell
sub DrawHints hr, hc
  local row, col, n, sg, x, y, hx, hy
  
  x = hmargin + (hc-1)*CSIZE+2
  y = vmargin + (hr-1)*CSIZE+2 
  sg = GetSubgridIndex(hr, hc)
  for i = 1 to SIZE
    n = 1
    if NumInRow(hr, i) then n = 0
    if NumInCol(hc, i) then n = 0
    if NumInSubgrid(sg, i) then n = 0
    if n = 1 then
      hx = x + ((i-1) mod 3)*HDX + 2
      hy = y + ((i-1)\3)*HDY + 2
      text hx, hy, str$(i), "LT", 7, 1, RGB(150, 150, 255), CCOLOR
    end if
  next i
end sub

' Draw the wait clock during automated puzzle solving
sub DrawWaitClock which
  local float a
  local cx, cy, x, y, c
  static float prev_a = 0
  cx = WCLOCKX - WCLOCKR
  cy = vmargin + SIZE*CSIZE\2
  if which = 1 then
    c = RGB(WHITE)
  else
    c = RGB(BLACK)
  end if
  circle cx, cy, WCLOCKR,,, c
  x = cx + int(0.8*cos(rad(prev_a))*WCLOCKR)
  y = cy - int(0.8*sin(rad(prev_a))*WCLOCKR)
  line cx, cy, x, y,, RGB(BLACK)
  a = prev_a - 15
  prev_a = a
  x = cx + int(0.8*cos(rad(a))*WCLOCKR)
  y = cy - int(0.8*sin(rad(a))*WCLOCKR)
  line cx, cy, x, y,, c
end sub
        
' reset the puzzle
sub ResetPuzzle
  local row, col
  for row = 1 to SIZE
    for col = 1 to SIZE
      if board(row, col) < 100 then
        board(row, col) = 0
      end if
    next col
  next row
  DrawPuzzle 1, 1
end sub

' Given a row, return 1 if the specified number exists in that row
function NumInRow(row, num)
  local i, hit, v
  hit = 0
  for i = 1 to SIZE
    v = board(row, i) mod 100
    if v = num then
      hit = 1
      exit for
    end if
  next i
  NumInRow = hit
end function

' Given a column, return 1 if the specified number exists in that column
function NumInCol(col, num)
  local i, hit, v
  hit = 0
  for i = 1 to SIZE
    v = board(i, col) mod 100
    if v = num then
      hit = 1
      exit for
    end if
  next i
  NumInCol = hit
end function

' Given a subgrid index, return 1 if the specified number exists in that subgrid
function NumInSubgrid(sgi, num)
  local sr, er, sc, ec, row, col, hit, v
  sr = ((sgi-1)\3)*3 + 1
  er = sr+2
  sc = ((sgi-1) mod 3)*3 + 1
  ec = sc+2
  hit = 0
  for row = sr to er
    for col = sc to ec
      v = board(row, col) mod 100
      if v = num then
        hit = 1
        exit for
      end if
    next col
    if hit = 1 then exit for
  next row
  NumInSubgrid = hit
end function

' Given a row and column, return the subgrid number that contains that cell
' Subgrids are numbered from 1 to SIZE in a raster fashion, top row to
' bottom and left to right in a row.
function GetSubgridIndex(row, col)
  local sga, sgb, sgi
  if row >= 1 and row <= ORDER then
    sga = 0
  else if row >= ORDER+1 and row <= 2*ORDER then
    sga = 3
  else
    sga = 6
  end if
  if col >= 1 and col <= ORDER then
    sgb = 1
  else if col >= ORDER+1 and col <= 2*ORDER then
    sgb = 2
  else
    sgb = 3
  end if
  sgi = sga+sgb
  GetSubgridIndex = sgi
end function

' Given a subgrid index, return the starting row and starting column indices
' The ending rows and cols are just the starting values +2.
sub GetSubgridStartRowAndCol sg, srow, scol
  srow = 1 + ((sg-1)\ORDER)*ORDER 
  scol = 1 + ((sg-1) mod ORDER)*ORDER
end sub

' Get list of possible numbers for the designated cell
sub GetNumberList row, col, nums()
  local n, i, hit, sg, nr, nc, ns
  n = 0
  if board(row, col) > 0 then
    nums(1) = 0
    exit sub
  end if
  sg = GetSubgridIndex(row, col)
  for i = 1 to SIZE
    hit = 1
    nr = NumInRow(row, i)
    nc = NumInCol(col, i)
    ns = NumInSubgrid(sg, i)
    if nr+nc+ns > 0 then hit = 0
    if hit = 1 then
      n = n+1
      nums(n+1) = i
    end if
  next i
  nums(1) = n
end sub

' Find trival solutions for cells where
' only one number is possible, add those
' numbers to the board. For best results,
' call this subroutine iteratively until
' count = zero. If push = 1, push moves
' onto the stack.
sub FindResidualNumbers count
  local row, col, sg
  local sr, er, sc, ec, n, t, nc, hit
  local nums(SIZE+1)
  local rnums(SIZE*SIZE, 3)

  ' look for residual number
  count = 0
  for row = 1 to SIZE
    for col = 1 to SIZE
      if board(row, col) = 0 then
        GetNumberList row, col, nums()
        if nums(1) = 1 then
          count = count+1
          rnums(count, 1) = row : rnums(count, 2) = col : rnums(count, 3) = nums(2)
        end if
      end if
    next col
  next row
  for i = 1 to count
    row = rnums(i, 1) : col = rnums(i, 2) : value = rnums(i, 3)
    board(row, col) = value
  next i
end sub

' Find numbers in subgrids that can be filled when other open cells
' are covered in rows and columns. If push = 1 then push moves onto move stack.
sub FindSubgridNumbers count
  local sg, row, col, sr, er, sc, ec, n
  local nrows, ncols, mrow  ncol
  local sgrows(ORDER), sgcols(ORDER)
  local rnums(SIZE*SIZE, 3)
  count = 0
  for sg = 1 to SIZE
    GetSubgridStartRowAndCol sg, sr, sc
    er = sr+2 : ec = sc+2
    for n = 1 to SIZE
      if not NumInSubgrid(sg, n) then
        nrows = 0
        for row = sr to er
          if NumInRow(row, n) then
            sgrows(row-sr+1) = n
            nrows = nrows+1
          else
            mrow = row
          end if
        next row
        if nrows = 2 then
          ncols = 0
          for col = sc to ec
            if board(mrow, col) > 0 then
              ncols = ncols+1
            else
              mcol = col
            end if
          next col
          if ncols = 2 then
            count = count+1
            rnums(count, 1) = mrow : rnums(count, 2) = mcol : rnums(count, 3) = n
          end if
        end if
        ncols = 0
        for col = sc to ec
          if NumInCol(col, n) then
            sgcols(col-sc+1) = n
            ncols = ncols+1
          else
            mcol = col
          end if
        next col
        if ncols = 2 then
          nrows = 0
          for row = sr to er
            if board(row, mcol) > 0 then
              nrows = nrows+1
            else
              mrow = row
            end if
          next row
          if nrows = 2 then
            count = count+1
            rnums(count, 1) = mrow : rnums(count, 2) = mcol : rnums(count, 3) = n
          end if
        end if
      end if
    next n
  next sg
for i = 1 to count
  row = rnums(i, 1) : col = rnums(i, 2) : value = rnums(i, 3)
  board(rnums(i, 1), rnums(i, 2)) = rnums(i,3)
next i
end sub

' Find cells in rows, columns, and subgrids where a number only
' occurs in a single cell, even when that cell has other choices.
' These occurances are trivial solutions.
' Unfortunately this algorithm is O(n**3) so a bit slow.
sub FindUniqueNumbers count
  local row, col, n, c, i, sr, sc, srow, scol, sg
  local nums(SIZE)
  local rnums(SIZE*SIZE, 3)
  count = 0
  for row = 1 to SIZE
    for n = 1 to SIZE
      c = 0
      for col = 1 to SIZE
        for i = 2 to nums(1)+1
          if nums(i) = n then 
            c = c+1
            scol = col
          end if
        next i
      next col
      if c = 1 then
        count = count+1 
        rnums(count, 1) = row : rnums(count, 2) = scol : rnums(count, 3) = n
      end if
    next n
    DrawWaitClock 1
  next row  
  for col = 1 to SIZE
    for n = 1 to SIZE
      c = 0
      for row = 1 to SIZE
        GetNumberList row, col, nums()
        for i = 2 to nums(1)+1
          if nums(i) = n then 
            c = c+1
            srow = row
          end if
        next i
      next row
      if c = 1 then
        count = count+1 
        rnums(count, 1) = srow : rnums(count, 2) = col : rnums(count, 3) = n
      end if
    next n
    DrawWaitClock 1
  next col
  for sg = 1 to SIZE
    GetSubgridStartRowAndCol sg, sr, sc
    for n = 1 to SIZE
      c = 0
      for row = sr to sr+2
        for col = sc to sc+2
          GetNumberList row, col, nums()
          for i = 2 to nums(1)+1
            if nums(i) = n then 
              c = c+1
              srow = row
              scol = col
            end if
          next i
        next col
      next row
      if c = 1 then
        count = count+1
        rnums(count, 1) = srow : rnums(count, 2) = scol : rnums(count, 3) = n
      end if
    next n
    DrawWaitClock 1
  next sg
  if count > 0 then
    for i = 1 to count
      row = rnums(i, 1) : col = rnums(i, 2) : n = rnums(i, 3)
      board(row, col) = n
    next i
  end if
end sub

' return a list of cells that have only a binary choice of numbers
' the return array has 4 values per cell: row, col, num1, num2
sub GetBinaryChoiceList n, cells()
  local row, col, num1, num2
  local nums(SIZE)
  n = 0
  for row = 1 to SIZE
    for col = 1 to SIZE
      GetNumberList row, col, nums()
      if nums(1) = 2 then
        n = n+1
        cells(n, 1) = row : cells(n, 2) = col
        cells(n, 3) = nums(2) : cells(n, 4) = nums(3)
      end if
    next col
  next row
end sub

' push a solving step into the solve stack
' components are guess tier, row, col, value
sub pushStep tier, row, col, value
  solve_stack(sptr, 1) = tier
  solve_stack(sptr, 2) = row
  solve_stack(sptr, 3) = col
  solve_stack(sptr, 4) = value
  solve_sptr = solve_sptr+1
end sub

' pop a solving step off the stack
sub popStep
  local tier row, col, value  
  
  solve_sptr = solve_sptr-1
  tier = solve_stack(sptr, 1)
  row = solve_stack(sptr, 2)
  col = solve_stack(sptr, 3)
  value = solve_stack(sptr, 4)
  board(row, col) = value
end sub
   
' returns 1 iff a valid solution has been found
function CheckWin()
  local row, col, sg, i, ok
  ok = 1  
  for i = 1 to SIZE
    for row = 1 to SIZE
      if not NumInRow(row, i)  then 
        ok = 0
        exit for
      end if
    next row
    if ok = 0 then
      CheckWin = 0
      exit function
    end if
    for col = 1 to SIZE
      if not NumInCol(col, i) then
        ok = 0
        exit for
      end if
    next col
    if ok = 0 then
      CheckWin = 0
      exit function
    end if
    for sg = 1 to SIZE
      if not NumInSubgrid(sg, i) then
        ok = 0
        exit for
      end if
      if ok = 0 then
        CheckWin = 0
        exit function
      end if
    next sg
  next i
  CheckWin = ok
end function

' returns true if the specified value placement
' would result in a collision or an unfillable cell.
' must be called BEFORE placement for accurate results.
function CheckConflict(row, col, num)
  local conflict = 0
  local sg, crow, ccol, sr, sc, sv
  local nums(SIZE+1)
  if num = 0 then exit function
  if NumInRow(row, num) then conflict = 1
  if NumInCol(col, num) then conflict = 1
  sg = GetSubgridIndex(row, col)
  if NumInSubgrid(sg, num) then conflict = 1
  if conflict then
    CheckConflict = 1
    exit function
  end if
  sv = board(row, col) 
  board(row, col) = num
  for scol = 1 to SIZE
    if board(row, scol) = 0 then
      GetNumberList row, scol, nums()
      if nums(1) = 0 then
        conflict = 1
        exit for
      end if
    end if
  next scol
  for srow = 1 to SIZE
    if board(srow, col) = 0 then
      GetNumberList srow, col, nums()
      if nums(1) = 0 then
        conflict = 1
        exit for
      end if
    end if
  next srow
  GetSubgridStartRowAndCol sg, sr, sc
  for srow = sr to sr+2
    for scol = sc to sc+2
      if board(srow, scol) = 0 then
        GetNumberList srow, scol, nums()
        if nums(1) = 0 then
          conflict = 1
          exit for
        end if
      end if
    next scol
    if conflict = 1 then exit for
  next srow
  board(row, col) = sv
  CheckConflict = conflict
end function
  
' show some instructions
sub ShowInstructions
  cls
  print "Sudoku is a game of pure logic. You fill cells in the 9 x 9 board"
  print "with numbers, but no arithmetic is involved. The cell contents could"
  print "be letters, meaningless symbols, colors, or even sounds. We use"
  print "numbers just because they are familiar."
  print ""
  print "The rules are very simple. A Sudoku puzzle is a 9 x 9 grid of squares."
  print "At the start, some of the squares have numbers and some are blank."
  print "The over all board is also divided into 9 'sub-grids', each of which"
  print "has a 3 x 3 subset of squares."
  print ""
  print "To solve a Sudoku puzzle, you have to fill in the blank squares with"
  print "numbers in the range 1 to 9. Those numbers have to honor this rule:"
  print "Every row, column, and sub-grid has to have exactly one of each of the"
  print "numbers from 1 to 9. None of them can have two of the same number."
  print ""
  print "That's it! Sudoku puzzles range from very easy to extremely difficult."
  print "They are graded Easy, Medium, Difficult, Hard, Evil, and Diabolical"
  print "in order of increasing difficulty.
  print ""
  print "Using the Menu, you can create your own puzzle, load an existing puzzle"
  print "from a file, save a puzzle to a file, and solve the puzzle yourself either
  print "with or without some computer assistance.
  print ""
  print "There are several sample Sudoku puzzles to help you get started. Use the"
  print "Load Puzzle command to choose a puzzle. For instance, choose Load Puzzle"
  print "from the Menu and then type in 'Easy/easy1' and press Enter. This will"
  print "load the specified puzzle and show it to you. 
  print ""
  print "To solve a puzzle, use the Arrow keys to navigate to a board cell and"
  print "press one of the number keys 1-9 to enter that number in the cell."
  print "You can use the zero ('0') key or the space bar to erase a number that"
  print "you previously entered. Note that you cannot modify or erase the original"
  print "filled cells; those are printed in blue."
  print ""
  print "To get hints, press 'H'. This will show you the possible values for each"
  print "blank cell. Press 'H' again to toggle the hints off.
  print "To get computer help in solving the puzzle, press 'S'. This will start an"
  print "automated puzzle solver. For easier puzzle, the automated solver will usually"
  print "be able to complete a solution. For harder puzzles, it can usually only get"
  print "partway to a solution, so you have to do the rest. You can use the 'S' command"
  print "as often as you want."
  print ""
  print "You can also make your own puzzle, or (more commonly) copy a puzzle from the"
  print "newspaper or book. Making a good Sudoku puzzle from scratch is very difficult!"
  print ""
  print "Press any Key to Continue..."
  z$ = INKEY$
  z$ = ""
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

sub ShowOptions
  local z$
  cls
  print "---------------------"
  print "     Options"
  print "---------------------"
  print ""
  print "Show hints while manually solving? (Y,N) [Default is Y: ";
  z$ = ""
  do
    z$ = INKEY$
  loop until z$ <> ""
  if LEFT$(UCASE$(z$), 1) = "N" then
    show_hints = 0
  else
    show_hints = 1
  end if
end sub
